/***************************************************************************//**
* \file SPIM_I2C.c
* \version 4.0
*
* \brief
*  This file provides the source code to the API for the SCB Component in
*  I2C mode.
*
* Note:
*
*******************************************************************************
* \copyright
* Copyright 2013-2017, Cypress Semiconductor Corporation.  All rights reserved.
* You may use this file only in accordance with the license, terms, conditions,
* disclaimers, and limitations in the end user license agreement accompanying
* the software package with which this file was provided.
*******************************************************************************/

#include "SPIM_PVT.h"
#include "SPIM_I2C_PVT.h"


/***************************************
*      I2C Private Vars
***************************************/

volatile uint8 SPIM_state;  /* Current state of I2C FSM */

#if(SPIM_SCB_MODE_UNCONFIG_CONST_CFG)

    /***************************************
    *  Configuration Structure Initialization
    ***************************************/

    /* Constant configuration of I2C */
    const SPIM_I2C_INIT_STRUCT SPIM_configI2C =
    {
        SPIM_I2C_MODE,
        SPIM_I2C_OVS_FACTOR_LOW,
        SPIM_I2C_OVS_FACTOR_HIGH,
        SPIM_I2C_MEDIAN_FILTER_ENABLE,
        SPIM_I2C_SLAVE_ADDRESS,
        SPIM_I2C_SLAVE_ADDRESS_MASK,
        SPIM_I2C_ACCEPT_ADDRESS,
        SPIM_I2C_WAKE_ENABLE,
        SPIM_I2C_BYTE_MODE_ENABLE,
        SPIM_I2C_DATA_RATE,
        SPIM_I2C_ACCEPT_GENERAL_CALL,
    };

    /*******************************************************************************
    * Function Name: SPIM_I2CInit
    ****************************************************************************//**
    *
    *
    *  Configures the SPIM for I2C operation.
    *
    *  This function is intended specifically to be used when the SPIM 
    *  configuration is set to “Unconfigured SPIM” in the customizer. 
    *  After initializing the SPIM in I2C mode using this function, 
    *  the component can be enabled using the SPIM_Start() or 
    * SPIM_Enable() function.
    *  This function uses a pointer to a structure that provides the configuration 
    *  settings. This structure contains the same information that would otherwise 
    *  be provided by the customizer settings.
    *
    *  \param config: pointer to a structure that contains the following list of 
    *   fields. These fields match the selections available in the customizer. 
    *   Refer to the customizer for further description of the settings.
    *
    *******************************************************************************/
    void SPIM_I2CInit(const SPIM_I2C_INIT_STRUCT *config)
    {
        uint32 medianFilter;
        uint32 locEnableWake;

        if(NULL == config)
        {
            CYASSERT(0u != 0u); /* Halt execution due to bad function parameter */
        }
        else
        {
            /* Configure pins */
            SPIM_SetPins(SPIM_SCB_MODE_I2C, SPIM_DUMMY_PARAM,
                                     SPIM_DUMMY_PARAM);

            /* Store internal configuration */
            SPIM_scbMode       = (uint8) SPIM_SCB_MODE_I2C;
            SPIM_scbEnableWake = (uint8) config->enableWake;
            SPIM_scbEnableIntr = (uint8) SPIM_SCB_IRQ_INTERNAL;

            SPIM_mode          = (uint8) config->mode;
            SPIM_acceptAddr    = (uint8) config->acceptAddr;

        #if (SPIM_CY_SCBIP_V0)
            /* Adjust SDA filter settings. Ticket ID#150521 */
            SPIM_SET_I2C_CFG_SDA_FILT_TRIM(SPIM_EC_AM_I2C_CFG_SDA_FILT_TRIM);
        #endif /* (SPIM_CY_SCBIP_V0) */

            /* Adjust AF and DF filter settings. Ticket ID#176179 */
            if (((SPIM_I2C_MODE_SLAVE != config->mode) &&
                 (config->dataRate <= SPIM_I2C_DATA_RATE_FS_MODE_MAX)) ||
                 (SPIM_I2C_MODE_SLAVE == config->mode))
            {
                /* AF = 1, DF = 0 */
                SPIM_I2C_CFG_ANALOG_FITER_ENABLE;
                medianFilter = SPIM_DIGITAL_FILTER_DISABLE;
            }
            else
            {
                /* AF = 0, DF = 1 */
                SPIM_I2C_CFG_ANALOG_FITER_DISABLE;
                medianFilter = SPIM_DIGITAL_FILTER_ENABLE;
            }

        #if (!SPIM_CY_SCBIP_V0)
            locEnableWake = (SPIM_I2C_MULTI_MASTER_SLAVE) ? (0u) : (config->enableWake);
        #else
            locEnableWake = config->enableWake;
        #endif /* (!SPIM_CY_SCBIP_V0) */

            /* Configure I2C interface */
            SPIM_CTRL_REG     = SPIM_GET_CTRL_BYTE_MODE  (config->enableByteMode) |
                                            SPIM_GET_CTRL_ADDR_ACCEPT(config->acceptAddr)     |
                                            SPIM_GET_CTRL_EC_AM_MODE (locEnableWake);

            SPIM_I2C_CTRL_REG = SPIM_GET_I2C_CTRL_HIGH_PHASE_OVS(config->oversampleHigh) |
                    SPIM_GET_I2C_CTRL_LOW_PHASE_OVS (config->oversampleLow)                          |
                    SPIM_GET_I2C_CTRL_S_GENERAL_IGNORE((uint32)(0u == config->acceptGeneralAddr))    |
                    SPIM_GET_I2C_CTRL_SL_MSTR_MODE  (config->mode);

            /* Configure RX direction */
            SPIM_RX_CTRL_REG      = SPIM_GET_RX_CTRL_MEDIAN(medianFilter) |
                                                SPIM_I2C_RX_CTRL;
            SPIM_RX_FIFO_CTRL_REG = SPIM_CLEAR_REG;

            /* Set default address and mask */
            SPIM_RX_MATCH_REG    = ((SPIM_I2C_SLAVE) ?
                                                (SPIM_GET_I2C_8BIT_ADDRESS(config->slaveAddr) |
                                                 SPIM_GET_RX_MATCH_MASK(config->slaveAddrMask)) :
                                                (SPIM_CLEAR_REG));


            /* Configure TX direction */
            SPIM_TX_CTRL_REG      = SPIM_I2C_TX_CTRL;
            SPIM_TX_FIFO_CTRL_REG = SPIM_CLEAR_REG;

            /* Configure interrupt with I2C handler but do not enable it */
            CyIntDisable    (SPIM_ISR_NUMBER);
            CyIntSetPriority(SPIM_ISR_NUMBER, SPIM_ISR_PRIORITY);
            (void) CyIntSetVector(SPIM_ISR_NUMBER, &SPIM_I2C_ISR);

            /* Configure interrupt sources */
        #if(!SPIM_CY_SCBIP_V1)
            SPIM_INTR_SPI_EC_MASK_REG = SPIM_NO_INTR_SOURCES;
        #endif /* (!SPIM_CY_SCBIP_V1) */

            SPIM_INTR_I2C_EC_MASK_REG = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_RX_MASK_REG     = SPIM_NO_INTR_SOURCES;
            SPIM_INTR_TX_MASK_REG     = SPIM_NO_INTR_SOURCES;

            SPIM_INTR_SLAVE_MASK_REG  = ((SPIM_I2C_SLAVE) ?
                            (SPIM_GET_INTR_SLAVE_I2C_GENERAL(config->acceptGeneralAddr) |
                             SPIM_I2C_INTR_SLAVE_MASK) : (SPIM_CLEAR_REG));

            SPIM_INTR_MASTER_MASK_REG = SPIM_NO_INTR_SOURCES;

            /* Configure global variables */
            SPIM_state = SPIM_I2C_FSM_IDLE;

            /* Internal slave variables */
            SPIM_slStatus        = 0u;
            SPIM_slRdBufIndex    = 0u;
            SPIM_slWrBufIndex    = 0u;
            SPIM_slOverFlowCount = 0u;

            /* Internal master variables */
            SPIM_mstrStatus     = 0u;
            SPIM_mstrRdBufIndex = 0u;
            SPIM_mstrWrBufIndex = 0u;
        }
    }

#else

    /*******************************************************************************
    * Function Name: SPIM_I2CInit
    ****************************************************************************//**
    *
    *  Configures the SCB for the I2C operation.
    *
    *******************************************************************************/
    void SPIM_I2CInit(void)
    {
    #if(SPIM_CY_SCBIP_V0)
        /* Adjust SDA filter settings. Ticket ID#150521 */
        SPIM_SET_I2C_CFG_SDA_FILT_TRIM(SPIM_EC_AM_I2C_CFG_SDA_FILT_TRIM);
    #endif /* (SPIM_CY_SCBIP_V0) */

        /* Adjust AF and DF filter settings. Ticket ID#176179 */
        SPIM_I2C_CFG_ANALOG_FITER_ENABLE_ADJ;

        /* Configure I2C interface */
        SPIM_CTRL_REG     = SPIM_I2C_DEFAULT_CTRL;
        SPIM_I2C_CTRL_REG = SPIM_I2C_DEFAULT_I2C_CTRL;

        /* Configure RX direction */
        SPIM_RX_CTRL_REG      = SPIM_I2C_DEFAULT_RX_CTRL;
        SPIM_RX_FIFO_CTRL_REG = SPIM_I2C_DEFAULT_RX_FIFO_CTRL;

        /* Set default address and mask */
        SPIM_RX_MATCH_REG     = SPIM_I2C_DEFAULT_RX_MATCH;

        /* Configure TX direction */
        SPIM_TX_CTRL_REG      = SPIM_I2C_DEFAULT_TX_CTRL;
        SPIM_TX_FIFO_CTRL_REG = SPIM_I2C_DEFAULT_TX_FIFO_CTRL;

        /* Configure interrupt with I2C handler but do not enable it */
        CyIntDisable    (SPIM_ISR_NUMBER);
        CyIntSetPriority(SPIM_ISR_NUMBER, SPIM_ISR_PRIORITY);
    #if(!SPIM_I2C_EXTERN_INTR_HANDLER)
        (void) CyIntSetVector(SPIM_ISR_NUMBER, &SPIM_I2C_ISR);
    #endif /* (SPIM_I2C_EXTERN_INTR_HANDLER) */

        /* Configure interrupt sources */
    #if(!SPIM_CY_SCBIP_V1)
        SPIM_INTR_SPI_EC_MASK_REG = SPIM_I2C_DEFAULT_INTR_SPI_EC_MASK;
    #endif /* (!SPIM_CY_SCBIP_V1) */

        SPIM_INTR_I2C_EC_MASK_REG = SPIM_I2C_DEFAULT_INTR_I2C_EC_MASK;
        SPIM_INTR_SLAVE_MASK_REG  = SPIM_I2C_DEFAULT_INTR_SLAVE_MASK;
        SPIM_INTR_MASTER_MASK_REG = SPIM_I2C_DEFAULT_INTR_MASTER_MASK;
        SPIM_INTR_RX_MASK_REG     = SPIM_I2C_DEFAULT_INTR_RX_MASK;
        SPIM_INTR_TX_MASK_REG     = SPIM_I2C_DEFAULT_INTR_TX_MASK;

        /* Configure global variables */
        SPIM_state = SPIM_I2C_FSM_IDLE;

    #if(SPIM_I2C_SLAVE)
        /* Internal slave variable */
        SPIM_slStatus        = 0u;
        SPIM_slRdBufIndex    = 0u;
        SPIM_slWrBufIndex    = 0u;
        SPIM_slOverFlowCount = 0u;
    #endif /* (SPIM_I2C_SLAVE) */

    #if(SPIM_I2C_MASTER)
    /* Internal master variable */
        SPIM_mstrStatus     = 0u;
        SPIM_mstrRdBufIndex = 0u;
        SPIM_mstrWrBufIndex = 0u;
    #endif /* (SPIM_I2C_MASTER) */
    }
#endif /* (SPIM_SCB_MODE_UNCONFIG_CONST_CFG) */


/*******************************************************************************
* Function Name: SPIM_I2CStop
****************************************************************************//**
*
*  Resets the I2C FSM into the default state.
*
*******************************************************************************/
void SPIM_I2CStop(void)
{
    /* Clear command registers because they keep assigned value after IP block was disabled */
    SPIM_I2C_MASTER_CMD_REG = 0u;
    SPIM_I2C_SLAVE_CMD_REG  = 0u;
    
    SPIM_state = SPIM_I2C_FSM_IDLE;
}


/*******************************************************************************
* Function Name: SPIM_I2CFwBlockReset
****************************************************************************//**
*
* Resets the scb IP block and I2C into the known state.
*
*******************************************************************************/
void SPIM_I2CFwBlockReset(void)
{
    /* Disable scb IP: stop respond to I2C traffic */
    SPIM_CTRL_REG &= (uint32) ~SPIM_CTRL_ENABLED;

    /* Clear command registers they are not cleared after scb IP is disabled */
    SPIM_I2C_MASTER_CMD_REG = 0u;
    SPIM_I2C_SLAVE_CMD_REG  = 0u;

    SPIM_DISABLE_AUTO_DATA;

    SPIM_SetTxInterruptMode(SPIM_NO_INTR_SOURCES);
    SPIM_SetRxInterruptMode(SPIM_NO_INTR_SOURCES);
    
#if(SPIM_CY_SCBIP_V0)
    /* Clear interrupt sources as they are not cleared after scb IP is disabled */
    SPIM_ClearTxInterruptSource    (SPIM_INTR_TX_ALL);
    SPIM_ClearRxInterruptSource    (SPIM_INTR_RX_ALL);
    SPIM_ClearSlaveInterruptSource (SPIM_INTR_SLAVE_ALL);
    SPIM_ClearMasterInterruptSource(SPIM_INTR_MASTER_ALL);
#endif /* (SPIM_CY_SCBIP_V0) */

    SPIM_state = SPIM_I2C_FSM_IDLE;

    /* Enable scb IP: start respond to I2C traffic */
    SPIM_CTRL_REG |= (uint32) SPIM_CTRL_ENABLED;
}


#if(SPIM_I2C_WAKE_ENABLE_CONST)
    /*******************************************************************************
    * Function Name: SPIM_I2CSaveConfig
    ****************************************************************************//**
    *
    *  Enables SPIM_INTR_I2C_EC_WAKE_UP interrupt source. This interrupt
    *  triggers on address match and wakes up device.
    *
    *******************************************************************************/
    void SPIM_I2CSaveConfig(void)
    {
    #if (!SPIM_CY_SCBIP_V0)
        #if (SPIM_I2C_MULTI_MASTER_SLAVE_CONST && SPIM_I2C_WAKE_ENABLE_CONST)
            /* Enable externally clocked address match if it was not enabled before.
            * This applicable only for Multi-Master-Slave. Ticket ID#192742 */
            if (0u == (SPIM_CTRL_REG & SPIM_CTRL_EC_AM_MODE))
            {
                /* Enable external address match logic */
                SPIM_Stop();
                SPIM_CTRL_REG |= SPIM_CTRL_EC_AM_MODE;
                SPIM_Enable();
            }
        #endif /* (SPIM_I2C_MULTI_MASTER_SLAVE_CONST) */

        #if (SPIM_SCB_CLK_INTERNAL)
            /* Disable clock to internal address match logic. Ticket ID#187931 */
            SPIM_SCBCLK_Stop();
        #endif /* (SPIM_SCB_CLK_INTERNAL) */
    #endif /* (!SPIM_CY_SCBIP_V0) */

        SPIM_SetI2CExtClkInterruptMode(SPIM_INTR_I2C_EC_WAKE_UP);
    }


    /*******************************************************************************
    * Function Name: SPIM_I2CRestoreConfig
    ****************************************************************************//**
    *
    *  Disables SPIM_INTR_I2C_EC_WAKE_UP interrupt source. This interrupt
    *  triggers on address match and wakes up device.
    *
    *******************************************************************************/
    void SPIM_I2CRestoreConfig(void)
    {
        /* Disable wakeup interrupt on address match */
        SPIM_SetI2CExtClkInterruptMode(SPIM_NO_INTR_SOURCES);

    #if (!SPIM_CY_SCBIP_V0)
        #if (SPIM_SCB_CLK_INTERNAL)
            /* Enable clock to internal address match logic. Ticket ID#187931 */
            SPIM_SCBCLK_Start();
        #endif /* (SPIM_SCB_CLK_INTERNAL) */
    #endif /* (!SPIM_CY_SCBIP_V0) */
    }
#endif /* (SPIM_I2C_WAKE_ENABLE_CONST) */


/* [] END OF FILE */
